<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';

// Define all possible permissions for this module
$productPermissions = [
    'view_products',
    'manage_products', 
    'add_product',
    'edit_product',
    'delete_product',
    'view_inventory',
    'manage_inventory',
    'add_stock',
    'edit_stock',
    'delete_stock'
];

// Check if user can access this page (has ANY product-related permission)
if (!hasAnyPermission($productPermissions)) {
    header('Location: dashboard.php');
    exit;
}

// Get user's specific permissions for this module
$userProductPermissions = getModulePermissions('product');
$userInventoryPermissions = getModulePermissions('inventory');

$activePage = 'products';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Granular Permissions Example</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Product Management</h1>
                    
                    <!-- Add Product Button - Only show if user has add_product permission -->
                    <?php if (hasPermission('add_product')): ?>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addProductModal">
                        <i class="fas fa-plus me-2"></i>Add Product
                    </button>
                    <?php else: ?>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>You don't have permission to add products.
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Products Section - Show if user can view products -->
                <?php if (hasPermission('view_products')): ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Products List</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Price</th>
                                        <th>Stock</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <!-- Sample data -->
                                    <tr>
                                        <td>1</td>
                                        <td>Sample Product</td>
                                        <td>$10.00</td>
                                        <td>50</td>
                                        <td>
                                            <!-- Edit button - Only show if user has edit_product permission -->
                                            <?php if (hasPermission('edit_product')): ?>
                                            <button class="btn btn-sm btn-primary" title="Edit Product">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <?php endif; ?>
                                            
                                            <!-- Delete button - Only show if user has delete_product permission -->
                                            <?php if (hasPermission('delete_product')): ?>
                                            <button class="btn btn-sm btn-danger" title="Delete Product">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                            <?php endif; ?>
                                            
                                            <!-- If user has no edit/delete permissions, show view-only message -->
                                            <?php if (!hasPermission('edit_product') && !hasPermission('delete_product')): ?>
                                            <span class="text-muted">View Only</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>You don't have permission to view products list.
                </div>
                <?php endif; ?>

                <!-- Inventory Section - Show if user can view inventory -->
                <?php if (hasPermission('view_inventory')): ?>
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Inventory Management</h5>
                        
                        <!-- Add Stock Button - Only show if user has add_stock permission -->
                        <?php if (hasPermission('add_stock')): ?>
                        <button type="button" class="btn btn-success btn-sm">
                            <i class="fas fa-plus me-1"></i>Add Stock
                        </button>
                        <?php endif; ?>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Stock Levels</h6>
                                <ul class="list-group">
                                    <li class="list-group-item d-flex justify-content-between align-items-center">
                                        Sample Product
                                        <span class="badge bg-primary rounded-pill">50</span>
                                    </li>
                                </ul>
                            </div>
                            
                            <!-- Stock Adjustment Section - Only show if user has edit_stock permission -->
                            <?php if (hasPermission('edit_stock')): ?>
                            <div class="col-md-6">
                                <h6>Quick Stock Adjustment</h6>
                                <div class="input-group">
                                    <input type="number" class="form-control" placeholder="Quantity">
                                    <button class="btn btn-outline-secondary" type="button">Adjust</button>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>You don't have permission to view inventory.
                </div>
                <?php endif; ?>

                <!-- Permissions Summary Card -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Your Permissions Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Product Permissions</h6>
                                <ul class="list-group list-group-flush">
                                    <?php 
                                    $productPerms = ['view_products', 'add_product', 'edit_product', 'delete_product'];
                                    foreach ($productPerms as $perm): 
                                    ?>
                                    <li class="list-group-item d-flex justify-content-between align-items-center px-0">
                                        <?php echo ucfirst(str_replace('_', ' ', $perm)); ?>
                                        <?php if (hasPermission($perm)): ?>
                                            <span class="badge bg-success">✓ Allowed</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">✗ Denied</span>
                                        <?php endif; ?>
                                    </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                            <div class="col-md-6">
                                <h6>Inventory Permissions</h6>
                                <ul class="list-group list-group-flush">
                                    <?php 
                                    $inventoryPerms = ['view_inventory', 'add_stock', 'edit_stock', 'delete_stock'];
                                    foreach ($inventoryPerms as $perm): 
                                    ?>
                                    <li class="list-group-item d-flex justify-content-between align-items-center px-0">
                                        <?php echo ucfirst(str_replace('_', ' ', $perm)); ?>
                                        <?php if (hasPermission($perm)): ?>
                                            <span class="badge bg-success">✓ Allowed</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">✗ Denied</span>
                                        <?php endif; ?>
                                    </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 